/*******************************************************************
 *
 * FileName:
 *	  VFSMark.c
 *
 * Copyright info:
 *        Copyright (c) 2001, Kopsis, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version with the following
 * additional restriction:
 *
 * 1) VFSMark(tm) is a trademark of Kopsis, Inc. You may not
 * distribute modified versions of this software that utilize the term
 * VFSMark in the name, documentation, or in any information displayed
 * by the program except when citing that your code is derived from
 * VFSMark.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Log: VFSMark.c,v $
 * Revision 1.3  2001/10/31 00:54:03  dkessler
 * Added ability to iterate through VFS volumes and let the user select
 * the one they want to test.
 *
 * Revision 1.2  2001/10/26 18:51:55  dkessler
 * Fixed volume size check to look for 1MiB instead of 10MiB.
 *
 * Revision 1.1  2001/10/25 16:34:52  dkessler
 * Initial revision
 *
 *******************************************************************/

#ifdef ERROR_CHECK_LEVEL
#undef ERROR_CHECK_LEVEL
#endif
#define ERROR_CHECK_LEVEL ERROR_CHECK_FULL

#define BUILDING_AGAINST_PALMOS35 1

#include <PalmOS.h>
#include <HsExt.h>
#include <CallbackFix.h>
#include "VFSMgr.h"
#include "VFSMarkRsc.h"

#define AppCreator 'vfsM'

// Test parameters

#define TEST_DIR "/PALM/Programs/VFSMark"
#define TEST_FILE "/PALM/Programs/VFSMark/test_0"
#define TEST_RECDB_PATH "/PALM/Programs/VFSMark/test_rec_db.pdb"
#define TEST_RESDB_PATH "/PALM/Programs/VFSMark/test_res_db.prc"
#define CREATE_TEST_COUNT 100
#define WRITE_TEST_COUNT1 1000
#define WRITE_TEST_COUNT2 100
#define WRITE_TEST_TOTAL (WRITE_TEST_COUNT1 + WRITE_TEST_COUNT2)
#define WRITE_TEST_SIZE1 16
#define WRITE_TEST_SIZE2 16384
#define SEEK_TEST_COUNT 20
#define NUM_TEST_RECS 50
#define TEST_REC_SIZE 4096
#define REC_TEST_COUNT 10
#define RANDOM_SEQUENCE_SIZE 10

#define ERR_DISK_FULL 2
#define ERR_GENERIC 3


// typedefs

typedef struct {
  UInt32 fileCreate;
  UInt32 fileDelete;
  UInt32 fileWrite;
  UInt32 fileRead;
  UInt32 fileSeek;
  UInt32 dbExport;
  UInt32 dbImport;
  UInt32 recAccess;
  UInt32 resAccess;
} Results_t;


// globals

// VFS volume under test
UInt16 vfsVolume_g;

// buffer used for read/write tests
MemHandle bufH_g = 0;
UInt8 *bufP_g = NULL;

// test results
Results_t vfsMark = {
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
  0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
  0xffffffff
};

// baseline Palm m500 test results
Results_t m500 = {
  2889, 680, 3085, 464, 118, 212, 763, 362, 384
};

// result strings
char results_g[10][16];

// state vars for popup windows
WinHandle progwin_g = 0;
WinHandle oldwin_g = 0;
WinHandle savewin_g = 0;

// sequence of semi-random values in the range 0 - 50
UInt16 random50_g[10] = {
  6, 11, 36, 12, 23, 16, 11, 47, 38, 5
};

// sequence of semi-random values in the range 0 - 200000
UInt32 random200k_g[10] = {
  124000, 45000, 16400, 187200, 104200,
  148800, 138600, 400, 120800, 196800
};

// timer variable
UInt32 startTicks_g;


// prototypes

static void mainFormUpdate(FormPtr frmP);


// functions

/***********************************************************
 *
 * Function: startTimer
 *
 * Description: Record the current tick count so that stopTimer can
 * compute the elapsed time.
 */
static void startTimer(void)
{
  startTicks_g = TimGetTicks();
}

/***********************************************************
 *
 * Function: stopTimer
 *
 * Description: Compute the elapsed time (in ticks) since the last
 * call to startTimer. This function can handle tick counter rollover.
 */
static UInt32 stopTimer(void)
{
  UInt32 stopTicks;

  stopTicks = TimGetTicks();
  if (stopTicks > startTicks_g) {
    return (stopTicks - startTicks_g);
  }
  else {
    return ((0xffffffff - startTicks_g) + stopTicks);
  }
}

/***********************************************************
 *
 * Function: drawProgressWindow
 *
 * Description: Create a popup window with a message and a progress
 * bar.
 */
static void drawProgressWindow(const Char *tag)
{
  Err err;
  RectangleType savebox = { {8, 38}, {144, 38} };
  RectangleType winbox = { {10, 40}, {140, 34} };
  RectangleType barbox = { {10, 19}, {120, 8} };
  FontID oldfont;

  savewin_g = WinSaveBits(&savebox, &err);
  progwin_g = WinCreateWindow(&winbox, dialogFrame, true, false, &err);
  if (progwin_g) {
    oldwin_g = WinGetDrawWindow();
    WinSetDrawWindow(progwin_g);
    WinEraseWindow();
    WinDrawWindowFrame();
    oldfont = FntGetFont();
    FntSetFont(stdFont);
    WinDrawTruncChars(tag, StrLen(tag), 5, 3, 120);
    FntSetFont(oldfont);
    WinDrawRectangleFrame(simpleFrame, &barbox);
  }
}


/***********************************************************
 *
 * Function: deleteProgressWindow
 *
 * Description: Remove the popup window and restore the screen.
 */
static void deleteProgressWindow(void)
{
  if (progwin_g) {
    WinDeleteWindow(progwin_g, true);
    WinSetDrawWindow(oldwin_g);
    progwin_g = 0;
  }
  if (savewin_g) {
    WinRestoreBits(savewin_g, 8, 38);
    savewin_g = 0;
  }
}


/***********************************************************
 *
 * Function: drawProgressBar
 *
 * Description: Draw a percent bar on the progress window.
 */
static void drawProgressBar(UInt16 percent)
{
  RectangleType barbox = { {10, 19}, {0, 8} };
  UInt32 size;

  size = (120 * (UInt32)percent) / 100;
  barbox.extent.x = size;
  WinDrawRectangle(&barbox, 0);
}


/***********************************************************
 *
 * Function: drawMessageWindow
 *
 * Description: Draw a popup up a message window for 2 seconds.
 */
static void drawMessageWindow(char *msg)
{
  drawProgressWindow(msg);
  SysTaskDelay(SysTicksPerSecond() * 2);
  deleteProgressWindow();
}


/***********************************************************
 *
 * Function: getObjectPtr
 *
 * Description: Return an object pointer give a form and an index.
 */
static void *getObjectPtr(FormPtr frmP, UInt16 index)
{
  UInt16 fldIndex =
      FrmGetObjectIndex(frmP, index);
  return FrmGetObjectPtr(frmP,
      fldIndex);
}

/***********************************************************
 *
 * Function: openTestDir
 *
 * Description: Open the test directory. If the directory does not
 * exist, it is created. If requested, the existing test dir and its
 * contents will be deleted first.
 */
static Err openTestDir(FileRef *dirRef, Boolean recreate)
{
  Err err;
  UInt32 iterator;
  FileInfoType info;
  char path[512];

  err = VFSFileOpen(vfsVolume_g, TEST_DIR, vfsModeRead, dirRef);
  if (!err) {
    if (recreate) {
      // we need to delete and recreate the directory just to make
      // make sure we get consistant results
      info.nameP = MemPtrNew(256);
      info.nameBufLen = 256;
      iterator = vfsIteratorStart;
      do {
        err = VFSDirEntryEnumerate(*dirRef, &iterator, &info);
        if (!err) {
          StrCopy(path, TEST_DIR);
          StrCat(path, "/");
          StrCat(path, info.nameP);
          //          drawMessageWindow(info.nameP);
          VFSFileDelete(vfsVolume_g, path);
        }
      } while (iterator != vfsIteratorStop);

      if (info.nameP) {
        MemPtrFree(info.nameP);
      }
      //      drawMessageWindow("closing dir");
      VFSFileClose(*dirRef);

      //      drawMessageWindow("deleting dir");
      VFSFileDelete(vfsVolume_g, TEST_DIR);
      err = vfsErrFileNotFound;
    }
    else {
      err = 0;
    }
  }

  if (err == vfsErrFileNotFound) {
    //    drawMessageWindow("constructing dir");
    err = VFSDirCreate(vfsVolume_g, "/PALM");
    if (!err || err == vfsErrFileAlreadyExists) {
      err = VFSDirCreate(vfsVolume_g, "/PALM/Programs");
      if (!err || err == vfsErrFileAlreadyExists) {
        err = VFSDirCreate(vfsVolume_g, "/PALM/Programs/VFSMark");
        if (!err) {
          err = VFSFileOpen(vfsVolume_g, TEST_DIR, vfsModeRead, dirRef);
        }
      }
    }
  }

  return err;
}

/***********************************************************
 *
 * Function: closeTestDir
 *
 * Description: Close the test directory.
 */
static Err closeTestDir(FileRef dirRef)
{
  Err err;

  // Right now we just close the dir. Later we may want to do some
  // tricks to make sure that caches and buffers get flushed.
  err = VFSFileClose(dirRef);

  return err;
}

/***********************************************************
 *
 * Function: createFile
 *
 * Description: Create a file with the given size (up to 16KB).
 */
static Err createFile(Char *path, UInt32 size)
{
  Err err;
  Err err2 = 0;
  FileRef fileRef;
  UInt32 numBytes;
  char msg[32];

  // File create can be faster if done in one step but that is broken
  // on PiVFS for the MemPlug
  err = VFSFileCreate(vfsVolume_g, path);
  if (err) {
    deleteProgressWindow();
    StrPrintF(msg, "err: %x", err);
    drawMessageWindow(msg);
  }

  err = VFSFileOpen(vfsVolume_g, path,
                    vfsModeReadWrite,
                    &fileRef);
  if (err) {
    deleteProgressWindow();
    StrPrintF(msg, "err: %x", err);
    drawMessageWindow(msg);
  }

  if (!err && size > 0) {
    // Write some random data to the file
    numBytes = size;
    err = VFSFileWrite(fileRef, numBytes, bufP_g, &numBytes);
    if (err) {
      deleteProgressWindow();
      StrPrintF(msg, "err: %x", err);
      drawMessageWindow(msg);
    }
    err2 = VFSFileClose(fileRef);
    if (err2) {
      deleteProgressWindow();
      StrPrintF(msg, "err: %x", err2);
      drawMessageWindow(msg);
    }
  }

  if (numBytes < size) {
    FrmAlert(DISK_FULL_ALERT);
    return ERR_DISK_FULL;
  }

  if (err || err2) {
    FrmAlert(TEST_FAIL_ALERT);
    return ERR_GENERIC;
  }
  else {
    return 0;
  }
}

/***********************************************************
 *
 * Function: doFileCreateTest
 *
 * Description: Test creating a large number of files.
 */
static UInt32 doFileCreateTest(void)
{
  UInt32 endTime;
  Err err;
  Char path[128];
  UInt16 i;
  FileRef dirRef;
  UInt16 iter = 0;
  UInt16 percent;

  // open test dir
  err = openTestDir(&dirRef, true);
  if (err) {
    return 0;
  }

  drawProgressWindow("File Create Test");

  startTimer();

  // create files
  for (i = 0; i < CREATE_TEST_COUNT; i++) {
    iter++;
    percent = (i * 100) / CREATE_TEST_COUNT;
    if (percent % 10 == 0) {
      drawProgressBar((i * 100) / CREATE_TEST_COUNT);
    }
    StrPrintF(path, "%s/test_%d", TEST_DIR, i);
    err = createFile(path, 9000);
    if (err) {
      deleteProgressWindow();
      return 0;
    }
  }

  endTime = stopTimer();

  deleteProgressWindow();

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  return (endTime);
}

/***********************************************************
 *
 * Function: doFileDeleteTest
 *
 * Description: Test deleting a large number of files.
 */
static UInt32 doFileDeleteTest(void)
{
  UInt32 endTime;
  Err err;
  Char path[128];
  UInt16 i;
  FileRef dirRef;
  UInt16 iter = 0;
  UInt16 percent;

  // open test dir
  err = openTestDir(&dirRef, false);
  if (err) {
    return 0;
  }

  drawProgressWindow("File Delete Test");

  startTimer();

  // delete files
  for (i = 0; i < CREATE_TEST_COUNT; i++) {
    iter++;
    percent = (i * 100) / CREATE_TEST_COUNT;
    if (percent % 10 == 0) {
      drawProgressBar((i * 100) / CREATE_TEST_COUNT);
    }
    StrPrintF(path, "%s/test_%d", TEST_DIR, i);
    err = VFSFileDelete(vfsVolume_g, path);
    if (err) {
      deleteProgressWindow();
      return 0;
    }
  }

  endTime = stopTimer();

  deleteProgressWindow();

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  return (endTime);
}

/***********************************************************
 *
 * Function: doFileReadTest
 *
 * Description: Test file reads. Does a mix of small and large
 * reads. This must be called after doFileWriteTest so the test file
 * exists.
 */
static UInt32 doFileReadTest(void)
{
  UInt32 endTime;
  Err err;
  FileRef dirRef;
  FileRef file0;
  UInt16 i;
  Boolean testPassed = true;
  UInt16 percent;

  // open test dir
  err = openTestDir(&dirRef, false);
  if (err) {
    return 0;
  }

  // open test file
  err = VFSFileOpen(vfsVolume_g, TEST_FILE,
                    vfsModeRead, &file0);
  if (err) {
    return 0;
  }
  
  drawProgressWindow("File Read Test");

  startTimer();

  // do small reads
  for (i = 0; i < WRITE_TEST_COUNT1; i++) {
    percent = ((UInt32)i * 100L) / WRITE_TEST_COUNT1;
    if (percent % 10 == 0) {
      percent = percent / 2;
      drawProgressBar(percent);
    }
    err = VFSFileRead(file0, WRITE_TEST_SIZE1, bufP_g, NULL);
    if (err) {
      testPassed = false;
    }
  }

  // do large reads
  for (i = 0; i < WRITE_TEST_COUNT2; i++) {
    percent = ((UInt32)i * 100L) / WRITE_TEST_COUNT2;
    drawProgressBar(50 + percent/2);
    err = VFSFileRead(file0, WRITE_TEST_SIZE2, bufP_g, NULL);
    if (err) {
      testPassed = false;
    }
  }

  endTime = stopTimer();

  deleteProgressWindow();

  VFSFileClose(file0);

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  if (testPassed) {
    return (endTime);
  }
  else {
    return 0;
  }
}

/***********************************************************
 *
 * Function: doFileWriteTest
 *
 * Description: Test file writes. Does a mix of small and large
 * writes.  
 */
static UInt32 doFileWriteTest(void)
{
  UInt32 endTime;
  Err err;
  FileRef dirRef;
  FileRef file0;
  UInt16 i;
  Boolean testPassed = true;
  UInt16 percent;

  // open test dir
  err = openTestDir(&dirRef, true);
  if (err) {
    return 0;
  }

  // create the test file
  err = VFSFileCreate(vfsVolume_g, TEST_FILE);
  if (err) {
    return 0;
  }

  err = VFSFileOpen(vfsVolume_g, TEST_FILE,
                    vfsModeReadWrite, &file0);
  if (err) {
    return 0;
  }
  
  drawProgressWindow("File Write Test");

  startTimer();

  // do small writes
  for (i = 0; i < WRITE_TEST_COUNT1; i++) {
    percent = ((UInt32)i * 100L) / WRITE_TEST_COUNT1;
    if (percent % 10 == 0) {
      percent = percent / 2;
      drawProgressBar(percent);
    }
    err = VFSFileWrite(file0, WRITE_TEST_SIZE1, bufP_g, NULL);
    if (err) {
      testPassed = false;
    }
  }

  // do large writes
  for (i = 0; i < WRITE_TEST_COUNT2; i++) {
    percent = ((UInt32)i * 100L) / WRITE_TEST_COUNT2;
    drawProgressBar(50 + percent/2);
    err = VFSFileWrite(file0, WRITE_TEST_SIZE2, bufP_g, NULL);
    if (err) {
      testPassed = false;
    }
  }

  endTime = stopTimer();

  deleteProgressWindow();

  VFSFileClose(file0);

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  if (testPassed) {
    return (endTime);
  }
  else {
    return 0;
  }
}

/***********************************************************
 *
 * Function: doFileSeekTest
 *
 * Description: Test a series of seeks to semi-random spots in a
 * file. This must be called after doWriteTest and doReadTest since it
 * uses the same test file and deletes it when finished.
 */
static UInt32 doFileSeekTest(void)
{
  UInt32 endTime;
  Err err;
  FileRef dirRef;
  FileRef file0;
  UInt16 i;
  UInt16 j;

  // open test dir
  err = openTestDir(&dirRef, false);
  if (err) {
    return 0;
  }

  err = VFSFileOpen(vfsVolume_g, TEST_FILE,
                    vfsModeRead, &file0);
  if (err) {
    return 0;
  }
  
  drawProgressWindow("File Seek Test");

  startTimer();

  for (i = 0; i < SEEK_TEST_COUNT; i++) {
    for (j = 0; j < RANDOM_SEQUENCE_SIZE; j++) {
      drawProgressBar((i * RANDOM_SEQUENCE_SIZE + j) / 2);
      err = VFSFileSeek(file0, vfsOriginBeginning, random200k_g[j]);
      VFSFileRead(file0, 16, bufP_g, NULL);
    }
  }

  endTime = stopTimer();

  deleteProgressWindow();

  VFSFileClose(file0);
  VFSFileDelete(vfsVolume_g, TEST_FILE);

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  return (endTime);
}

/***********************************************************
 *
 * Function: doDbExportTest
 *
 * Description: Create two large databases (~200KB) in RAM (one record
 * and one resource) and test exporting them to the card.
 */
static UInt32 doDbExportTest(void)
{
  UInt32 endTime;
  Err err;
  FileRef dirRef;
  LocalID resDbID;
  LocalID recDbID;
  DmOpenRef dbRef;
  UInt16 recIndex;
  Boolean testFailed = false;
  UInt16 i;
  MemHandle recH;
  MemHandle resH;

  // open test dir
  err = openTestDir(&dirRef, true);
  if (err) {
    return 0;
  }

  drawProgressWindow("DB Export Test");

  // create a record database
  err = DmCreateDatabase(0, "VFSMarkRecDB", 'vfsM', 'data', false);
  recDbID = DmFindDatabase(0, "VFSMarkRecDB");
  if (!recDbID) {
    deleteProgressWindow();
    return 0;
  }
  dbRef = DmOpenDatabase(0, recDbID, dmModeReadWrite);
  if (!dbRef) {
    deleteProgressWindow();
    return 0;
  }

  // add records
  for (i = 0; i < NUM_TEST_RECS; i++) {
    recIndex = dmMaxRecordIndex;
    recH = DmNewRecord(dbRef, &recIndex, TEST_REC_SIZE);
    if (!recH) {
      testFailed = true;
      break;
    }
    DmReleaseRecord(dbRef, recIndex, true);
  }

  // close database
  DmCloseDatabase(dbRef);

  drawProgressBar(25);

  // create a resource database
  err = DmCreateDatabase(0, "VFSMarkResDB", 'vfsM', 'DATA', true);
  resDbID = DmFindDatabase(0, "VFSMarkResDB");
  if (!resDbID) {
    deleteProgressWindow();
    return 0;
  }
  dbRef = DmOpenDatabase(0, resDbID, dmModeReadWrite);
  if (!dbRef) {
    deleteProgressWindow();
    return 0;
  }

  // add resources
  for (i = 0; i < NUM_TEST_RECS; i++) {
    recIndex = dmMaxRecordIndex;
    resH = DmNewResource(dbRef, 'test', i, TEST_REC_SIZE);
    if (!resH) {
      testFailed = true;
      break;
    }
    DmReleaseResource(resH);
  }

  // close database
  DmCloseDatabase(dbRef);

  drawProgressBar(50);

  startTimer();

  err = VFSExportDatabaseToFile(vfsVolume_g, TEST_RECDB_PATH, 
                                0, recDbID);
  if (err) {
    testFailed = true;
  }

  drawProgressBar(75);

  err = VFSExportDatabaseToFile(vfsVolume_g, TEST_RESDB_PATH,
                                0, resDbID);
  if (err) {
    testFailed = true;
  }

  drawProgressBar(100);

  endTime = stopTimer();

  // delete record database
  DmDeleteDatabase(0, recDbID);

  // delete resource database
  DmDeleteDatabase(0, resDbID);

  deleteProgressWindow();

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  return (endTime);
}

/***********************************************************
 *
 * Function: doRecAccessTest
 *
 * Description: Test accessing records in a database on the card. This
 * test reads a semi-random sequence of records from the DB. This
 * function must be called after doExportTest since it uses the
 * exported database.  
 */
static UInt32 doRecAccessTest(void)
{
  UInt32 endTime;
  Err err;
  FileRef dirRef;
  FileRef file0;
  UInt16 i;
  UInt16 j;
  MemHandle recH;

  // open test dir
  err = openTestDir(&dirRef, false);
  if (err) {
    return 0;
  }

  // open test db
  err = VFSFileOpen(vfsVolume_g, TEST_RECDB_PATH, vfsModeRead, &file0);
  if (err) {
    return 0;
  }

  drawProgressWindow("Record Access Test");

  startTimer();

  for (i = 0; i < REC_TEST_COUNT; i++) {
    for (j = 0; j < RANDOM_SEQUENCE_SIZE; j++) {
      drawProgressBar(i * RANDOM_SEQUENCE_SIZE + j);
      err = VFSFileDBGetRecord(file0, random50_g[j], &recH, NULL, NULL);
      if (recH) {
        MemHandleFree(recH);
      }
    }
  }

  endTime = stopTimer();

  deleteProgressWindow();

  err = VFSFileClose(file0);

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  return (endTime);
}

/***********************************************************
 *
 * Function: doResAccessTest
 *
 * Description: Test accessing resources in a database on the
 * card. This test reads a semi-random sequence of resources from the
 * DB. This function must be called after doExportTest since it uses
 * the exported database.  
 */
static UInt32 doResAccessTest(void)
{
  UInt32 endTime;
  Err err;
  FileRef dirRef;
  FileRef file0;
  UInt16 i;
  UInt16 j;
  MemHandle resH;

  // open test dir
  err = openTestDir(&dirRef, false);
  if (err) {
    return 0;
  }

  // open test db
  err = VFSFileOpen(vfsVolume_g, TEST_RESDB_PATH, vfsModeRead, &file0);
  if (err) {
    return 0;
  }

  drawProgressWindow("Resource Access Test");

  startTimer();

  for (i = 0; i < REC_TEST_COUNT; i++) {
    for (j = 0; j < RANDOM_SEQUENCE_SIZE; j++) {
      drawProgressBar(i * RANDOM_SEQUENCE_SIZE + j);
      err = VFSFileDBGetResource(file0, 'test', random50_g[j], &resH);
      if (resH) {
        MemHandleFree(resH);
      }
    }
  }

  endTime = stopTimer();

  deleteProgressWindow();

  err = VFSFileClose(file0);

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  return (endTime);
}

/***********************************************************
 *
 * Function: doDbImportTest
 *
 * Description: Test importing databases from the card. This function
 * imports two large (200KB) databases from the card. This function
 * must be called after doDbExportTest since it uses the exported
 * databases. Since importing is pretty fast, the test is repeated to
 * get a more accurate timing.
 */
static UInt32 doDbImportTest(void)
{
  UInt32 endTime;
  Err err;
  FileRef dirRef;
  LocalID recDbID;
  LocalID resDbID;

  // open test dir
  err = openTestDir(&dirRef, false);
  if (err) {
    return 0;
  }

  drawProgressWindow("DB Import Test");

  // if the databases exist, delete them
  recDbID = DmFindDatabase(0, "VFSMarkRecDB");
  if (recDbID) {
    DmDeleteDatabase(0, recDbID);
  }

  resDbID = DmFindDatabase(0, "VFSMarkResDB");
  if (recDbID) {
    DmDeleteDatabase(0, resDbID);
  }

  startTimer();

  err = VFSImportDatabaseFromFile(vfsVolume_g, TEST_RECDB_PATH, 
                                  NULL, &recDbID);

  drawProgressBar(25);

  err = VFSImportDatabaseFromFile(vfsVolume_g, TEST_RESDB_PATH, 
                                  NULL, &resDbID);

  drawProgressBar(50);

  // if the databases exist, delete them
  if (recDbID) {
    DmDeleteDatabase(0, recDbID);
  }

  if (recDbID) {
    DmDeleteDatabase(0, resDbID);
  }

  err = VFSImportDatabaseFromFile(vfsVolume_g, TEST_RECDB_PATH, 
                                  NULL, &recDbID);

  drawProgressBar(75);

  err = VFSImportDatabaseFromFile(vfsVolume_g, TEST_RESDB_PATH, 
                                  NULL, &resDbID);

  drawProgressBar(100);

  endTime = stopTimer();

  // if the databases exist, delete them
  if (recDbID) {
    DmDeleteDatabase(0, recDbID);
  }

  if (recDbID) {
    DmDeleteDatabase(0, resDbID);
  }

  deleteProgressWindow();

  err = closeTestDir(dirRef);
  if (err) {
    return 0;
  }

  return (endTime);
}

/***********************************************************
 *
 * Function: doTestCleanup
 *
 * Description: Cleanup any stuff leftover from the tests.
 */
static void doTestCleanup(void)
{
  Err err;
  FileRef dirRef;

  // open test dir
  err = openTestDir(&dirRef, true);
  if (err) {
    return;
  }

  closeTestDir(dirRef);
}

/***********************************************************
 *
 * Function: runAllTests
 *
 * Description: Run all the VFS tests and record the results. The
 * displayed results are updated after each test. The order the tests
 * run in is critical! Don't change it unless you really understand
 * what you're doing.
 */
static Err runAllTests(FormPtr frmP)
{
  UInt32 time;

  // run all tests
  if (FrmAlert(CONFIRM_TESTS_ALERT) == 0) {
    return 0;
  }

  // file create
  time = doFileCreateTest();
  vfsMark.fileCreate = (time * 100) / SysTicksPerSecond();
  if (vfsMark.fileCreate == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // file delete
  time = doFileDeleteTest();
  vfsMark.fileDelete = (time * 100) / SysTicksPerSecond();
  if (vfsMark.fileDelete == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // file write
  time = doFileWriteTest();
  vfsMark.fileWrite = (time * 100) / SysTicksPerSecond();
  if (vfsMark.fileWrite == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // file read
  time = doFileReadTest();
  vfsMark.fileRead = (time * 100) / SysTicksPerSecond();
  if (vfsMark.fileRead == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // file seek
  time = doFileSeekTest();
  vfsMark.fileSeek = (time * 100) / SysTicksPerSecond();
  if (vfsMark.fileSeek == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // db export
  time = doDbExportTest();
  vfsMark.dbExport = (time * 100) / SysTicksPerSecond();
  if (vfsMark.dbExport == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // record access
  time = doRecAccessTest();
  vfsMark.recAccess = (time * 100) / SysTicksPerSecond();
  if (vfsMark.recAccess == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // resource access
  time = doResAccessTest();
  vfsMark.resAccess = (time * 100) / SysTicksPerSecond();
  if (vfsMark.resAccess == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  // db import
  time = doDbImportTest();
  vfsMark.dbImport = (time * 100) / SysTicksPerSecond();
  if (vfsMark.dbImport == 0) {
    // test failed
    return 1;
  }
  mainFormUpdate(frmP);

  doTestCleanup();

  //  EvtSetAutoOffTimer(SetAtLeast, 60);

  return 0;
}


/***********************************************************
 *
 * Function: computeVFSMark
 *
 * Description: Compute an average of the percentage scores (relative
 * to the Palm m500) for all the tests.
 */
static UInt32 computeVFSMark(void)
{
  UInt32 total = 0;

  total += (m500.fileCreate * 100L)/vfsMark.fileCreate;
  total += (m500.fileDelete * 100L)/vfsMark.fileDelete;
  total += (m500.fileWrite * 100L)/vfsMark.fileWrite;
  total += (m500.fileRead * 100L)/vfsMark.fileRead;
  total += (m500.fileSeek * 100L)/vfsMark.fileSeek;
  total += (m500.dbExport * 100L)/vfsMark.dbExport;
  total += (m500.dbImport * 100L)/vfsMark.dbImport;
  total += (m500.recAccess * 100L)/vfsMark.recAccess;
  total += (m500.resAccess * 100L)/vfsMark.resAccess;

  total = total / 9;

  return (total);
}

/***********************************************************
 *
 * Function: saveResults
 *
 * Description: Save all the test results in a Memo Pad memo.
 */
static void saveResults(void)
{
  DmOpenRef dbRef = 0;
  MemHandle recH = 0;
  char *memo = NULL;
  char *memoP = NULL;
  UInt8 *recP = NULL;
  UInt16 recIndex;

  // allocate space for a memo on dynamic heap
  memo = MemPtrNew(4096);
  if (!memo) {
    goto Exit;
  }

  // dump everthing into a big long string
  memoP = memo;
  StrCopy(memoP, "VFSMark Results\n\n");
  memoP += StrLen(memo);

  // would be nice to include some system info here

  StrPrintF(memoP, "File Create:\t%ld%%\n", 
	    (m500.fileCreate * 100L)/vfsMark.fileCreate);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "File Delete:\t%ld%%\n", 
	    (m500.fileDelete * 100L)/vfsMark.fileDelete);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "File Write:\t%ld%%\n", 
	    (m500.fileWrite * 100L)/vfsMark.fileWrite);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "File Read:\t%ld%%\n", 
	    (m500.fileRead * 100L)/vfsMark.fileRead);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "File Seek:\t%ld%%\n", 
	    (m500.fileSeek * 100L)/vfsMark.fileSeek);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "DB Export:\t%ld%%\n", 
	    (m500.dbExport * 100L)/vfsMark.dbExport);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "DB Import:\t%ld%%\n", 
	    (m500.dbImport * 100L)/vfsMark.dbImport);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "Record Access:\t%ld%%\n", 
	    (m500.recAccess * 100L)/vfsMark.recAccess);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "Resource Access:\t%ld%%\n", 
	    (m500.resAccess * 100L)/vfsMark.resAccess);
  memoP += StrLen(memoP);
  StrPrintF(memoP, "\nVFSMark:\t%ld\n", computeVFSMark());

  // open the memo database
  dbRef = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
  if (!dbRef) {
    goto Exit;
  }

  // create a new memo record
  recIndex = dmMaxRecordIndex;
  recH = DmNewRecord(dbRef, &recIndex, StrLen(memo) + 1);
  if (!recH) {
    goto Exit;
  }

  // copy the string to the record
  recP = MemHandleLock(recH);
  DmWrite(recP, 0, memo, StrLen(memo) + 1);
  MemHandleUnlock(recH);

  FrmAlert(EXPORT_DONE_ALERT);
  
 Exit:

  if (recH) {
    DmReleaseRecord(dbRef, recIndex, true);
  }

  if (dbRef) {
    DmCloseDatabase(dbRef);
  }

  if (memo) {
    MemPtrFree(memo);
  }
}

/***********************************************************
 *
 * Function: resultsFieldsUpdate
 *
 * Description: Update the text of all the results fields.
 */
static void resultsFieldsUpdate(FormPtr frmP)
{
  // initialize all the test result fields
#ifndef RAW_RESULTS
  StrPrintF(results_g[0], "%ld%%", (m500.fileCreate * 100L)/vfsMark.fileCreate);
  StrPrintF(results_g[1], "%ld%%", (m500.fileDelete * 100L)/vfsMark.fileDelete);
  StrPrintF(results_g[2], "%ld%%", (m500.fileWrite * 100L)/vfsMark.fileWrite);
  StrPrintF(results_g[3], "%ld%%", (m500.fileRead * 100L)/vfsMark.fileRead);
  StrPrintF(results_g[4], "%ld%%", (m500.fileSeek * 100L)/vfsMark.fileSeek);
  StrPrintF(results_g[5], "%ld%%", (m500.dbExport * 100L)/vfsMark.dbExport);
  StrPrintF(results_g[6], "%ld%%", (m500.dbImport * 100L)/vfsMark.dbImport);
  StrPrintF(results_g[7], "%ld%%", (m500.recAccess * 100L)/vfsMark.recAccess);
  StrPrintF(results_g[8], "%ld%%", (m500.resAccess * 100L)/vfsMark.resAccess);
  StrPrintF(results_g[9], "%ld", computeVFSMark());
#else
  StrPrintF(results_g[0], "%ld", vfsMark.fileCreate);
  StrPrintF(results_g[1], "%ld", vfsMark.fileDelete);
  StrPrintF(results_g[2], "%ld", vfsMark.fileWrite);
  StrPrintF(results_g[3], "%ld", vfsMark.fileRead);
  StrPrintF(results_g[4], "%ld", vfsMark.fileSeek);
  StrPrintF(results_g[5], "%ld", vfsMark.dbExport);
  StrPrintF(results_g[6], "%ld", vfsMark.dbImport);
  StrPrintF(results_g[7], "%ld", vfsMark.recAccess);
  StrPrintF(results_g[8], "%ld", vfsMark.resAccess);
  StrPrintF(results_g[9], "%ld", computeVFSMark());
#endif

  FldSetTextPtr(getObjectPtr(frmP, FILE_CREATE_FLD), results_g[0]);
  FldSetTextPtr(getObjectPtr(frmP, FILE_DELETE_FLD), results_g[1]);
  FldSetTextPtr(getObjectPtr(frmP, FILE_WRITE_FLD), results_g[2]);
  FldSetTextPtr(getObjectPtr(frmP, FILE_READ_FLD), results_g[3]);
  FldSetTextPtr(getObjectPtr(frmP, FILE_SEEK_FLD), results_g[4]);
  FldSetTextPtr(getObjectPtr(frmP, DB_EXPORT_FLD), results_g[5]);
  FldSetTextPtr(getObjectPtr(frmP, DB_IMPORT_FLD), results_g[6]);
  FldSetTextPtr(getObjectPtr(frmP, REC_ACCESS_FLD), results_g[7]);
  FldSetTextPtr(getObjectPtr(frmP, RES_ACCESS_FLD), results_g[8]);
  FldSetTextPtr(getObjectPtr(frmP, VFSMARK_FLD), results_g[9]);
}

/***********************************************************
 *
 * Function: mainFormUpdate
 *
 * Description: Redraw all the fields on the main form (after updating
 * them).
 */
static void mainFormUpdate(FormPtr frmP)
{
  resultsFieldsUpdate(frmP);

  FldDrawField(getObjectPtr(frmP, FILE_CREATE_FLD));
  FldDrawField(getObjectPtr(frmP, FILE_DELETE_FLD));
  FldDrawField(getObjectPtr(frmP, FILE_WRITE_FLD));
  FldDrawField(getObjectPtr(frmP, FILE_READ_FLD));
  FldDrawField(getObjectPtr(frmP, FILE_SEEK_FLD));
  FldDrawField(getObjectPtr(frmP, DB_EXPORT_FLD));
  FldDrawField(getObjectPtr(frmP, DB_IMPORT_FLD));
  FldDrawField(getObjectPtr(frmP, REC_ACCESS_FLD));
  FldDrawField(getObjectPtr(frmP, RES_ACCESS_FLD));
  FldDrawField(getObjectPtr(frmP, VFSMARK_FLD));
}

/***********************************************************
 *
 * Function: mainFormInit
 *
 * Description: Initialize the main form.
 */
static void mainFormInit(FormPtr frmP)
{
  FormPtr frmP2 = FrmGetActiveForm();

  resultsFieldsUpdate(frmP2);

  FrmDrawForm(frmP2);
}

/***********************************************************
 *
 * Function: doMenu
 *
 * Description: Handle menu items.
 */
static Boolean doMenu(FormPtr frmP, UInt16 command)
{
  Boolean handled = false;
  switch (command) {
  case ABOUT_MENU:
    FrmAlert(AboutAlert);
    handled = true;
    break;
  case TEST_MENU:
    runAllTests(frmP);
    handled = true;
    break;
  case SAVE_MENU:
    handled = true;
    break;
  case INFO_MENU:
    FrmAlert(INFO_ALERT);
    handled = true;
    break;
  }
  return handled;
}

/***********************************************************
 *
 * Function: mainFormEventHandler
 *
 * Description: Handle main form events (menus and buttons).
 */
Boolean mainFormEventHandler(EventPtr eventP)
{
  Boolean handled = false;
  FormPtr frmP = FrmGetActiveForm();
  switch (eventP->eType) {
  case frmOpenEvent:
    FrmDrawForm(frmP);
    mainFormInit(frmP);
    handled = true;
    break;
  case menuEvent:
    handled = doMenu(frmP,
        eventP->data.menu.itemID);
    break;
  case ctlSelectEvent: 
    switch (eventP->data.ctlSelect.controlID) {
    case RUN_TESTS_BTN :
      runAllTests(frmP);
      handled = true;
      break;
    case SAVE_RESULTS_BTN :
      saveResults();
      handled = true;
      break;
    case INFO_BTN :
      FrmAlert(INFO_ALERT);
      handled = true;
      break;
    default:
      break;
    }
  default:
    break;
  }
  return handled;
}

/***********************************************************
 *
 * Function: startApp
 *
 * Description: Initialize globals and perform some checks to make
 * sure we can run the tests.
 */
static Err startApp()
{
  Err err;
  UInt32 vfsMgrVersion;
  UInt32 volIter;
  UInt32 volumeTotal;
  UInt32 volumeUsed;
  UInt32 freeRam;
  UInt16 vfsVolumes[8];
  UInt16 i;
  UInt16 j;
  VolumeInfoType volInfo;
  UInt16 selection;
  char mediaStr[20];
  
  // check for VFS
  err = FtrGet(sysFileCVFSMgr, vfsFtrIDVersion, &vfsMgrVersion); 
  if(err){ 
    // VFS Manager not installed
    FrmAlert(NO_VFS_ALERT);
    return 1;
  }

  // get first volume
  volIter = vfsIteratorStart;
  i = 0;
  while (i < 8) {
    err = VFSVolumeEnumerate(&(vfsVolumes[i++]), &volIter);
    if (err) {
      break;
    }
    if (volIter == vfsIteratorStop) {
      break;
    }
  }

  if (i == 0) {
    // no VFS volumes
    return 1;
  }

  // let the user pick a volume
  for (j = 0; j < i; j++) {
    err = VFSVolumeInfo(vfsVolumes[j], &volInfo);
    switch (volInfo.mediaType) {
    case expMediaType_MemoryStick:
      StrCopy(mediaStr, "A MemoryStick");
      break;
    case expMediaType_CompactFlash:
      StrCopy(mediaStr, "A CF card");
      break;
    case expMediaType_SecureDigital:
      StrCopy(mediaStr, "An SD card");
      break;
    case expMediaType_MultiMediaCard:
      StrCopy(mediaStr, "An MMC");
      break;
    case expMediaType_SmartMedia:
      StrCopy(mediaStr, "A SmartMedia card");
      break;
    case expMediaType_RAMDisk:
      StrCopy(mediaStr, "A RAM disk");
      break;
    case expMediaType_PoserHost:
    case expMediaType_MacSim:
      StrCopy(mediaStr, "An emulator disk");
      break;
    default:
      StrCopy(mediaStr, "An unknown");
      break;
    }
    selection = FrmCustomAlert(WHICH_VOLUME_ALERT, mediaStr, " ", " ");
    if (!selection) {
      break;
    }
  }

  if (j < i) {
    vfsVolume_g = vfsVolumes[j];
  }
  else {
    return 1;
  }

  // check space on flash card
  VFSVolumeSize(vfsVolume_g, &volumeUsed, &volumeTotal);
  if (volumeTotal - volumeUsed < 1000000) {
    FrmAlert(DISK_FULL_ALERT);
    return 1;
  }

  // check space in RAM
  MemCardInfo(0, NULL, NULL, NULL, NULL, NULL, NULL, &freeRam);
  if (freeRam < 500000) {
    FrmAlert(NO_MEM_ALERT);
  }

  // allocate buffer
  bufH_g = MemHandleNew(16L * 1024L);
  if (!bufH_g) {
    FrmAlert(NO_MEM_ALERT);
    return 1;
  }
  bufP_g = MemHandleLock(bufH_g);

  return 0;
}

/***********************************************************
 *
 * Function: stopApp
 *
 * Description: Clean up any globals holding allocated memory.
 */
static void stopApp()
{
  if (bufP_g) {
    MemHandleUnlock(bufH_g);
    bufP_g = NULL;
  }

  if (bufH_g) {
    MemHandleFree(bufH_g);
    bufH_g = 0;
  }
}

/***********************************************************
 *
 * Function: appHandleEvent
 *
 * Description: Generic app event handler.
 */
Boolean appHandleEvent(EventPtr event)
{
  FormPtr	frm;
  UInt16		formId;
  Boolean	handled = false;
  
  if (event->eType == frmLoadEvent) {
    formId =
        event->data.frmLoad.formID;
    frm = FrmInitForm(formId);
    FrmSetActiveForm(frm);
    if (formId == MainForm)
      FrmSetEventHandler (frm,
          mainFormEventHandler);
    handled = true;
  }	
  return handled;
}

/***********************************************************
 *
 * Function: PilotMain
 *
 * Description: Generic PilotMain.
 */
UInt32 PilotMain(UInt16 cmd, void *cmdPBP, UInt16 launchFlags)
{
  EventType event;
  Err error;
  
  if (cmd == sysAppLaunchCmdNormalLaunch) {

    if (!startApp()) {
      FrmGotoForm(MainForm);
      do {
        EvtGetEvent(&event, 30);
        if (!SysHandleEvent(&event)) {
          if (!MenuHandleEvent(0, &event, &error)) {
            if (!appHandleEvent(&event)) {
              FrmDispatchEvent(&event);
            }
          }
        }
      } while (event.eType != appStopEvent);
      stopApp();

      FrmCloseAllForms();
    }
  }
  return 0;
}
